/***********************************************************************************************************************
* File Name     : command.ino
* Version        : 1.0.0.0
* Device(s)     : Arduino UNO/DUE/...
* Description   : Pressure sensor command process File
***********************************************************************************************************************/
/**********************************************************************************************************************
* History : DD.MM.YYYY Version  Description
*         : 07.10.2024 1.0.0.0  First Release
***********************************************************************************************************************/
#define __COMMAND_C__

//==============================================================================
// INCLUDES
//==============================================================================
#include "command.h"
#include "config.h"

//==============================================================================
// VARIABLES
//==============================================================================
uint8_t data[12];

//==============================================================================
// Reset Function
//==============================================================================
void Command_Proc(void)
{
    uint8_t count;
    uint8_t command;
    uint8_t dataSize;
    
    if(Serial.available()>1){
        command = Serial.read();            // Read command code
        delay(1);    // wait read(1ms)

        switch(command){
            case IDE_CMD_START:             // for serial monitor
            case IDE_CMD_START2:            // for serial monitor
                IDE_Start();
                break;
            case IDE_CMD_STOP:              // for serial monitor
            case IDE_CMD_STOP2:             // for serial monitor
                IDE_Stop();
                break;
            case APP_CMD_INST:              // for application 
                dataSize = Serial.read();   // Read data length
                delay(1);                   // wait read(1ms)
                for(count=0; count<dataSize; count++){
                    data[count] = Serial.read();    // Read command data
                    delay(1);               // wait read(1ms)
                }

                // command processing
                switch(data[0]){
                    case APP_CMD_VERSION:
                        Command_ReadFwVersion();        // read firmware version
                        break;
                    case APP_CMD_START:
                        Command_StartActive(data);      // start active
                        break;
                    case APP_CMD_STOP:
                        Command_StopResult();           // stop result
                        break;
                    case APP_CMD_SELECT_BOARD:
                        Command_SelectMcuBoard();       // Select MCU board
                        break;
                    case APP_CMD_SET_SLA:
                        Command_SetSlaveAddress(data);  // Set slave address
                        break;
                    case APP_CMD_READ_PROTOCOL:
                        Command_ReadProtocol();         // Read protocol
                        break;
                    case APP_CMD_READ_MODEL:
                        Command_ReadModel();         // Read protocol
                        break;
                    default:                            // other command
                        data[0] = ERROR_CODE_NOT_SUPPORT;
                        data[1] = 0;
                        Serial.write(data, 2);
                }
            default:
                return;
        }
    }
}

//==============================================================================
// Select MCU Board Command
//==============================================================================
void Command_SelectMcuBoard(void)
{
    data[0] = ERROR_CODE_NO_ERROR;
    data[1] = 0;
    Serial.write(data, 2);
}

//==============================================================================
// Read Firmware Version Command
//==============================================================================
void Command_ReadFwVersion(void)
{
    standAloneFlag = false;  
    data[0] = ERROR_CODE_NO_ERROR;
    data[1] = 4;
    data[2] = sketch_version[0];
    data[3] = sketch_version[1];
    data[4] = sketch_version[2];
    data[5] = sketch_version[3];
    Serial.write(data, 6);
}

//==============================================================================
// Read Protocol Command
//==============================================================================
void Command_ReadProtocol(void)
{
    data[0] = ERROR_CODE_NO_ERROR;
    data[1] = 1;
    data[2] = psel;    // I2C:0  SPI:1
    Serial.write(data, 3);
}

//==============================================================================
// Read Sensor Model
//==============================================================================
void Command_ReadModel(void)
{
    standAloneFlag = false;  
    data[0] = ERROR_CODE_NO_ERROR;
    data[1] = 6;
    data[2] = model_name[0];
    data[3] = model_name[1];
    data[4] = model_name[2];
    data[5] = model_name[3];
    data[6] = model_name[4];
    data[7] = model_name[5];
    Serial.write(data, 8);
}

//==============================================================================
// Set SlaveAddress Command
//==============================================================================
void Command_SetSlaveAddress(uint8_t *data)
{
    if(data[1] < 0x80){
        slaveAddress = data[1];
        data[0] = ERROR_CODE_NO_ERROR;
        data[1] = 0;
    }
    else{
        data[0] = ERROR_CODE_ILLEGAL_PARAMETER;
        data[1] = 0;
    }
    Serial.write(data, 2);
}

//==============================================================================
// Start Active Command
//==============================================================================
void Command_StartActive(uint8_t *data)
{
    uint32_t temp;
    
    activeCmd = CMD_ACTIVE1 + (data[1]*2);      // active mode
    resultCmd = data[2];                        // result mode
    skipNum = data[3];                          // skip number
    skipCnt = skipNum;
    iir_coeffB[0] = data[4];                    // filter coefficient
    iir_coeffB[1] = data[5];
    iir_coeffB[2] = data[6];
    iir_coeffB[3] = data[7];
    temp  = ((uint32_t)data[8]) << 24;          // interval time
    temp += ((uint32_t)data[9]) << 16;
    temp += ((uint32_t)data[10]) << 8;
    temp += ((uint32_t)data[11]);
    waitTimeData = temp;

    if(waitTimeData == 0){
        DRDYB_measureFlag = true;
    }
    else{
        DRDYB_measureFlag = false;
    }
    
    resetReq = true;                   // Sensor reset
    macramWriteReq = true;             // Lpf setting
    startReq = true;                    // Sensor start
    standAloneFlag = false;             // for application
    
    if(resultCmd == 0){
        result1Flag = true;             // Pressure data read
        result2Flag = false;
        result3Flag = false;
        result4Flag = false;
    }
    else if(resultCmd == 1){
        result1Flag = false;
        result2Flag = true;             // Temperature data read
        result3Flag = false;
        result4Flag = false;
    }
    else if(resultCmd == 2){
        result1Flag = false;
        result2Flag = false;            // Pressure 1st lpf data read
        result3Flag = true;
        result4Flag = false;
    }
    else if(resultCmd == 3){
        result1Flag = false;
        result2Flag = false;
        result3Flag = false;
        result4Flag = true;             // Pressure 2nd lpf data read
    }
    else{
        result1Flag = true;
        result2Flag = false;
        result3Flag = false;            // Pressure data read
        result4Flag = false;
    }
    
    data[0] = ERROR_CODE_NO_ERROR;
    data[1] = 0;
    Serial.write(data, 2);              // Send responsedata 
}

//==============================================================================
// Start Result Command
//==============================================================================
void Command_StopResult(void)
{
    operateFlag = false;          // Sensor stop
    standAloneFlag = false;     // for application

    data[0] = ERROR_CODE_NO_ERROR;
    data[1] = 0;
    Serial.write(data, 2);
}

//==============================================================================
// IDE Start Result Command
//==============================================================================
void IDE_Start(void)
{
    waitTimeData = waitTime;
    resetReq = true;           // Sensor reset
    startReq = true;            // Sensor start
    standAloneFlag = true;      // Serial monitor
    if(result3Flag == true){
        macramWriteReq = true;
    }
}

//==============================================================================
// IDE Stop Result Command
//==============================================================================
void IDE_Stop(void)
{
    operateFlag = false;          // Sensor stop
    standAloneFlag = true;      // Serial mointor
}

//==============================================================================
// Create response data
//==============================================================================
void Create_ResponseData(uint8_t *data)
{   
    data[0] = ERROR_CODE_NO_ERROR;      // errorcode
    data[1] = reaponseDataLength;       // data length

    if(resultCmd == 0){
        data[2] = Result1Data[0];       // Pressure data
        data[3] = Result1Data[1];
        data[4] = Result1Data[2];
    }
    else if(resultCmd == 1){
        data[2] = Result2Data[0];       // temperature data
        data[3] = Result2Data[1];
        data[4] = Result2Data[2];
    }
    else if(resultCmd == 2){
        data[2] = Result3Data[0];       // pressure 1st Lpf data
        data[3] = Result3Data[1];
        data[4] = Result3Data[2];
    }
    else if(resultCmd == 3){
        data[2] = Result4Data[0];       // pressure 2nd Lpf data
        data[3] = Result4Data[1];
        data[4] = Result4Data[2];
    }
    else{
        data[2] = Result1Data[0];
        data[3] = Result1Data[1];
        data[4] = Result1Data[2];
    }
    data[5] = (uint8_t)(pastTime>>16);  // set interval time
    data[6] = (uint8_t)(pastTime>>8);
    data[7] = (uint8_t)(pastTime);
}

//==============================================================================
// Create response data
//==============================================================================
void OutputData()
{
    if(standAloneFlag == false){
        //----------
        // case of Evaluation Program
        //----------
        Create_ResponseData(response_data);         // create response data
        Serial.write(response_data, reaponseDataLength+2 );             // send response data
        waitTime = waitTimeData;                    // set interval
    
    }else{
        //----------
        // case of Arduino IDE(for Serial Monitor)
        //----------
        Serial.print(pastTime);             // Send interval data to serial monitor
        
        // Display on serial monitor
        if(result1Flag == true){
            IDE_Result1Data = convert_int32(Result1Data);           // Convert to int32 pressure Data 
            Serial.print(sep);                                      // Send log separator
            Serial.print(IDE_Result1Data);                          // Send pressure data to serial monitor
        }
        if(result2Flag == true){
            IDE_Result2Data = convert_int32(Result2Data);           // Convert to int32 temperature data
            Serial.print(sep);                                      // Send log separator
            Serial.print(IDE_Result2Data);                          // Send temperature data to serial monitor 
        }
        if(result3Flag == true){
            IDE_Result3Data = convert_int32(Result3Data);           // Convert to int32 pressure 1st lpf data
            Serial.print(sep);                                      // Send log separator
            Serial.print(IDE_Result3Data);                          // Send pressure 1st lpf data to serial monitor 
        }
        if(result4Flag == true){
            IDE_Result4Data = convert_int32(Result4Data);           // Convert to int32 pressure 2nd lpf data
            Serial.print(sep);                                      // Send log separator
            Serial.print(IDE_Result4Data);                          // Send pressure 2nd lpf data to serial monitor 
        }
        
        Serial.println("");
        
        waitTime = waitTimeData;          // set interval
    }
}
