/***********************************************************************************************************************
* File Name     : Mitsumi_Pressure_Sensor_SDK_for_Arduino_SPI_Sample_Sketch.ino
* Version        : 1.0.0.0
* Device(s)     : Arduino UNO/DUE/...
* Description   : Pressure sensor sample program (SPI)
***********************************************************************************************************************/
/**********************************************************************************************************************
* History : DD.MM.YYYY Version  Description
*         : 07.10.2024 1.0.0.0  First Release
***********************************************************************************************************************/
#define _MAIN_

//==============================================================================
// INCLUDES
//==============================================================================
#include <SPI.h>
#include "config.h"
#include "command.h"

//==============================================================================
// DEFINES AND MACROS
//==============================================================================
//#define DEBUG                        // for Debug (Stand alone only)

#define CMD_RESET           0x72    // Reset command
#define CMD_IDLE            0x94    // Idle command
#define CMD_WRITE_MACRAM    0xE4    // MACRAM Write command

// Pressure Sensor result command
#define CMD_RESULT1  0xC0           // Read Pressure Result
#define CMD_RESULT2  0xC2           // Read Temperature Result
#define CMD_RESULT3  0xC4           // Read Pressure 1st Lpf Result
#define CMD_RESULT4  0xC6           // Read Pressure 2nd Lpf Result

// Status Code
#define STC_OK        0x00          // No Error
#define STC_ERR1      0x01          // Communication error

//==============================================================================
// VARIABLES
//==============================================================================
// Timer
uint32_t startTimerCount;               // measuring start timer count
uint32_t nowTimerCount;                 // update timer count 
uint32_t pastTime;                      // interval
uint32_t waitTimeData;                  // set waittime from app
uint32_t TimeoutCount = 1000000;        // 1000ms timeout

// flag
boolean standAloneFlag = true;          // true="Serial monitor", false="Evaluation Progmram"
boolean startReq = false;               // measure start request
boolean operateFlag = false;            // operate flag
boolean resetReq = false;              // sensor reset request
boolean macramWriteReq = false;        // MACRAM write request

// Command data
uint8_t resultCmd;                      // set result cmmand from application
uint8_t skipCnt = 0;                    // skip counter
uint8_t slaveAddress = SLAVE_ADDR;      // set slave address from application
uint8_t iir_coeffB[4];                  // filter coeffcient
uint32_t lpfFcMACRAM;                   // Cutoff Frequency -> MACRAM setting(standAlone only)

// Result data
uint8_t Result1Data[3];                 // Pressure result data
uint8_t Result2Data[3];                 // Temperature result data
uint8_t Result3Data[3];                 // Pressure 1st Lpf result data
uint8_t Result4Data[3];                 // Pressure 2nd Lpf result data

// Serial monitor send data
int32_t IDE_Result1Data;                // Pressresult result data
int32_t IDE_Result2Data;                // Temperature result data
int32_t IDE_Result3Data;                // Pressure 1st Lpf result data
int32_t IDE_Result4Data;                // Pressure 2nd Lpf result data

uint8_t errorCode;                      // error status

//==============================================================================
// setup
//==============================================================================
void setup() {
    Serial.begin(115200);           // serial boudrate: 115200bps
    
    // Initialize ports
    pinMode(CSB, OUTPUT);
    digitalWrite(CSB, HIGH);
    pinMode(PSEL, OUTPUT);
    digitalWrite(PSEL, HIGH);       // PSEL Low:I2C  High:SPI
    pinMode(SPI_EN, OUTPUT);
    digitalWrite(SPI_EN, HIGH);     // SPI enable
    pinMode(I2C_EN, OUTPUT);
    digitalWrite(I2C_EN, LOW);      // I2C disenable
    pinMode(DRDYB, INPUT);          // DRDYB input

    // Initialize SPI
    SPI.begin();
    SPI.setDataMode(SPI_MODE3);             // SPI Mode:3
    SPI.setBitOrder(MSBFIRST);              // MSB first
    SPI.setClockDivider(SPI_CLOCK_DIV16);   // SCLK=1MHz

    // Initialize parameter
    errorCode = STC_OK;
    startReq = false;
    resetReq = false;
    macramWriteReq = false;
    operateFlag = false;

#ifdef DEBUG
    // Send Serial Monitor
    Serial.print("IntervalTime[usec]");
    if(result1Flag == true){
        Serial.print(sep);
        Serial.print("Presssure");
    }
    if(result2Flag == true){
        Serial.print(sep);
        Serial.print("Temperature");
    }
    if(result3Flag == true){
        Serial.print(sep);
        Serial.print("Presssure 1st Lpf");
    }
    if(result4Flag == true){
        Serial.print(sep);
        Serial.print("Presssure 2nd Lpf");
    }
    Serial.println("");
#endif
    
}

//==============================================================================
// loop
//==============================================================================
void loop() {

    // Command process
    Command_Proc();

    // Error process
    if(errorCode != STC_OK){
        return;
    }

    // Sensor reset
    if(resetReq == true){
        if(operateFlag == false){
            resetReq = false;
            sensor_reset();                         // Reset sensor
            delay(20);                              // wait 20ms
        }
    }

    // MACRAM write
    if(macramWriteReq == true){
        if(operateFlag == false){
            if(standAloneFlag == true){
                MACRAM_set();
            }
            macramWriteReq = false;
            sensor_idle();                          // Idle state
            delay(10);                              // wait 10ms
            write_MACRAM();                         // MACRAM Write filter coeffcient
            delay(15);                              // wait 15ms
        }
    }
    
    // Sensor start
    if(startReq == true){
        startReq = false;
        if(operateFlag == false){
            operateFlag = true;
            if(DRDYB_measureFlag == false){
                sensor_1st_waittime(activeCmd);  // 1st interval setting
            }
            startTimerCount = micros();             // Timer start 
            sensor_start(activeCmd);                // measurering start
        }
        else{
            // no process
        }
    }
    else{
        if(operateFlag == false){
            return;   
        }
    }
    
    // Check interval
    if(sensor_wait_conversion() == false){
        return;
    }
    
    // Read result
    if(result1Flag == true){
        sensor_read_result(CMD_RESULT1, Result1Data, 3);        // Read result pressure
    }
    if(result2Flag == true){
        sensor_read_result(CMD_RESULT2, Result2Data, 3);        // Read result temperature
    }
    if(result3Flag == true){
        sensor_read_result(CMD_RESULT3, Result3Data, 3);        // Read result pressure 1st Lpf
    }
    if(result4Flag == true){
        sensor_read_result(CMD_RESULT4, Result4Data, 3);        // Read result pressure 2nd Lpf
    }

    //----------
    // output data
    //----------
    OutputData();
}

//==============================================================================
// Reset Function
//==============================================================================
void sensor_reset(void)
{
    digitalWrite(CSB, LOW);
    SPI.transfer(CMD_RESET);
    digitalWrite(CSB, HIGH);
}

//==============================================================================
// Idle Function
//==============================================================================
void sensor_idle(void)
{
    digitalWrite(CSB, LOW);
    SPI.transfer(CMD_IDLE);
    digitalWrite(CSB, HIGH);
}

//==============================================================================
// Start Measurement Function
//==============================================================================
void sensor_start(uint8_t mode)
{
    digitalWrite(CSB, LOW);
    SPI.transfer(mode);
    digitalWrite(CSB, HIGH);
}

//==============================================================================
// Read Result Function
//==============================================================================
void sensor_read_result(uint8_t target, uint8_t *data, uint8_t dataLen)
{
    uint8_t cnt;

    digitalWrite(CSB, LOW);
    SPI.transfer(target);
    for(cnt=0; cnt<dataLen; cnt++){
        data[cnt] = SPI.transfer(0x00);
    }
    digitalWrite(CSB, HIGH);
}


//==============================================================================
// Convert Result Data Function
//==============================================================================
uint32_t convert_int32(uint8_t *result)
{
    int32_t data;

    data = (int32_t)result[2];
    data += (int32_t)result[1] * 256;
    data += (int32_t)result[0] * 256 * 256;
    data <<= 8;
    data >>= 8;

    return data;
}

//==============================================================================
// Setting for first conversion time Function
//==============================================================================
void sensor_1st_waittime(uint8_t mode)
{
    switch(mode){
        case CMD_ACTIVE1:
            waitTime = ACTIVE1_1ST_WAITTIME;        // Active1: wait(max.=3.1ms)
            break;
        case CMD_ACTIVE2:
            waitTime = ACTIVE2_1ST_WAITTIME;        // Active2: wait(max.=6.1ms)
            break;
        case CMD_ACTIVE3:
            waitTime = ACTIVE3_1ST_WAITTIME;        // Active3: wait(max.=12.2ms)
            break;
        case CMD_ACTIVE4:
            waitTime = ACTIVE4_1ST_WAITTIME;        // Active4: wait(max.=24.3ms)
            break;
        default:
            waitTime = ACTIVE1_1ST_WAITTIME;
            break;
    }
}

//==============================================================================
// Waiting for conversion Function
//==============================================================================
boolean sensor_wait_conversion()
{
    uint32_t waitTimeUs;
    
    //----------
    // DRDYB mode
    //----------
    if(DRDYB_measureFlag == true){
        // check DRDYB
        while(digitalRead(DRDYB) == LOW){
            // read now time
            nowTimerCount = micros();
            if(Check_Timeout(TimeoutCount) == false){            //1000ms timeout
                errorCode = STC_ERR1;
                return false;
            }
        }
        // check DRDYB
        while(digitalRead(DRDYB) != LOW){
            nowTimerCount = micros();
            if(Check_Timeout(TimeoutCount) == false){            //1000ms timeout
                errorCode = STC_ERR1;
                return false;
            }
        }

        // calculate the elapsed time
        if(nowTimerCount > startTimerCount){
            pastTime = nowTimerCount - startTimerCount;
        }else{
            pastTime = 0xFFFFFFFF - startTimerCount;
            pastTime += nowTimerCount;
        }

        // Check skip count
        if(skipCnt < skipNum){              // skip count check
            skipCnt++;
            return false;
        }
        skipCnt = 0;                        // skip count reset
        startTimerCount = nowTimerCount;    // update startTimerCount
        
        return true;
    }

    //----------
    // Timer mode
    //----------
    else{       
        // read now time
        nowTimerCount = micros();

        // calculate the elapsed time
        if(nowTimerCount > startTimerCount){
            pastTime = nowTimerCount - startTimerCount;
        }else{
            pastTime = 0xFFFFFFFF - startTimerCount;
            pastTime += nowTimerCount;
        }
       
        // check the elapsed time
        if(pastTime < waitTime){
            return false;
        }
       
        // update startTimerCount
        startTimerCount = nowTimerCount;
        return true;
    }
}

//==============================================================================
// MACRAM Setting
//==============================================================================
void MACRAM_set(){
    if(activeCmd == CMD_ACTIVE1){
        lpfFcMACRAM = ACTIVE1_LPFFC_MACRAM;
    }
    else if(activeCmd == CMD_ACTIVE2){
        lpfFcMACRAM = ACTIVE2_LPFFC_MACRAM;
    }
    else if(activeCmd == CMD_ACTIVE3){
        lpfFcMACRAM = ACTIVE3_LPFFC_MACRAM;
    }
    else{
        lpfFcMACRAM = ACTIVE4_LPFFC_MACRAM;
    }
    iir_coeffB[0] = (uint8_t)((lpfFcMACRAM >> 24) % 256);
    iir_coeffB[1] = (uint8_t)((lpfFcMACRAM >> 16) % 256);
    iir_coeffB[2] = (uint8_t)((lpfFcMACRAM >> 8) % 256);
    iir_coeffB[3] = (uint8_t)((lpfFcMACRAM ) % 256);
}

//==============================================================================
// MACRAM Write Command
//==============================================================================
void write_MACRAM(){
    uint8_t data[6];
    
    data[0] = CMD_WRITE_MACRAM;
    data[1] = 0x6F;
    data[2] = iir_coeffB[0];
    data[3] = iir_coeffB[1];
    data[4] = iir_coeffB[2];
    data[5] = iir_coeffB[3];
    
    digitalWrite(CSB, LOW);
    SPI.transfer(data,6);
    digitalWrite(CSB, HIGH);
}

//==============================================================================
// Check Timeout
//==============================================================================
boolean Check_Timeout(uint32_t limitTime){
    if(nowTimerCount > startTimerCount){
        pastTime = nowTimerCount - startTimerCount;
    }else{
        pastTime = 0xFFFFFFFF - startTimerCount;
        pastTime += nowTimerCount;
    }
    if (pastTime > limitTime){
        return false;
    }
    else{
        return true;
    }
}
